function [pointSize] = fun_pointsize(vertices, type, K)
% Copyright (C) 2019 ECOLE POLYTECHNIQUE FEDERALE DE LAUSANNE, Switzerland
%
%     Multimedia Signal Processing Group (MMSPG)
%
%     This program is free software: you can redistribute it and/or modify
%     it under the terms of the GNU General Public License as published by
%     the Free Software Foundation, either version 3 of the License, or
%     (at your option) any later version.
%
%     This program is distributed in the hope that it will be useful,
%     but WITHOUT ANY WARRANTY; without even the implied warranty of
%     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%     GNU General Public License for more details.
%
%     You should have received a copy of the GNU General Public License
%     along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%
% Author:
%   Evangelos Alexiou (evangelos.alexiou@epfl.ch)
%
% Reference:
%   Alexiou, E., Viola, I., Borges, T., Fonseca, T., De Queiroz, R., &
%   Ebrahimi, T. (2019). A comprehensive study of the rate-distortion
%   performance in MPEG point cloud compression. APSIPA Transactions on
%   Signal and Information Processing, 8, E27.
%
%
% Determine the size assigned to a point based on Euclidean distances
%   between this point and its K nearest neighbors. The point size can be
%   either fixed or adaptive. In the first case, the same value is used
%   across the whole model, whereas in the second case, a different value
%   is assigned to every point of the model based on local resolutions.
%
%   [pointSize] = fun_pointsize(vertices, type, K)
%
%   INPUTS
%       vertices:   Coordinates of a point cloud (e.g., ptCloud.Location).
%       splatType:  Splats of "fixed" or "adaptive" size.
%       K:          Number of nearest neighbors that define neighborhoods
%                   over which local statistics are computed.
%
%   OUTPUTS
%       pointSize:  The size of splats for a point cloud model. The indices
%                   follow the same order as every other attribute of the
%                   model.
%
%   Example:
%   [pointSize] = fun_pointsize(ptCloud.Location, 'adaptive', 12)


if nargin < 3
    error('Too few input arguments.');
elseif nargin == 3
    switch type
        case {'adaptive', 'fixed'}
        otherwise
            error('splatType can be either "fixed", or "adaptive".');
    end
end

% K-nearest neighbors
[~, d] = knnsearch(vertices, vertices, 'K', K+1, 'NSMethod', 'kdtree', 'Distance', 'euclidean');
d(:, 1) = [];

% Local statistics, per point
localMean = mean(d, 2);
localStd = std(d, [], 2);

% Global statistics, per model
globalMean = mean(localMean);
globalStd = mean(localStd);

% Identification of isolated points whose splat size is set as global mean
localMean(localMean >= globalMean + 3*globalStd | localMean <= globalMean - 3*globalStd) = globalMean;


if strcmp(type, 'fixed')
    pointSize = globalMean;
elseif strcmp(type, 'adaptive')
    pointSize = localMean;
else
    pointSize = [];
end
